{
    This file is part of the Turbo51 examples.
    Copyright (C) 2008 - 2011 by Igor Funa

    http://turbo51.com/

    This file is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
}

{$IDATA }

Program Example3;

Uses I2C, E24C32;

Type  Word_ = Record
                Case Boolean of
                  True:  (L: Byte; H: Byte);
                  False: (W: Word);
              end;

      LongInt_ = Record
                   Case Boolean of
                     True:  (WL: Word_; WH: Word_);
                     False: (L: LongInt);
                 end;


     TEEprom = Record
                 EE_Signature:             Word; {  2 Bytes }
                 EE_NumberOfLearnedKeysLo: Byte; {  1 Byte  }
                 EE_ModuleAddress:         Byte; {  1 Byte  }
                 EE_RelayTime:             Array [1..4] of Word_; {  4 * 2 Bytes }
                 EE_ModuleType:            Byte; {  1 byte  }
                 EE_ModuleFlags:           Byte; {  1 byte  }
                 EE_Spare1:                Word; {  2 bytes }
                 EE_ChannelConfig:         Array [1..15] of Byte; {  15 bytes }
                 EE_NumberOfLearnedKeysHi: Byte; {  1 byte  }
                 EE_TXKeyData:             Byte;
               end;

     TxData = Record
                tdKey:          Array [0..7] of Byte;
                tdSerialNumber: Array [0..3] of Byte;
                tdCounter:      Word_;
              end;

Const VersionHi = $01;
      VersionLo = $00;

      Signature = $FECA;

      Greeting = 'TURBO51'#0;
      GreetingString: Array [0..Length (Greeting) - 1] of Char = Greeting;

      ManufacturerKeyData: Array [0..7] of Byte = ($CA, $FE, $00, $00, $03, $3D, $B5, $CA);

      TimerConst = $C7AF;
      Const1ms   = $F8CD;

      BaudRateTimerValue = Word (- 22118400 div 32 div 9600);

      LedTime         =    30;
      RS485_Time      =     2;
      SetupTime       = 30000;
      NoKeyTime       =   100;

      DafaultRelayLongPulseTime  =   30000 div 32;
      DafaultLightPulseTime      = 1800000 div 32;
      DefaultRelayShortPulseTime =     384 div 32;
      DefaultRelayOffPulseTime   =    1500 div 32;

      Relay_ON  = LowLevel;
      Relay_OFF = HighLevel;

      MotorUp = 1;
      MotorDown = 0;

      Segments: Array [' '..'U'] of Byte = (
        %00000000, { ' ' }
        %00000000, { '!' }
        %00100010, { '"' }
        %00000000, { '#' }
        %00000000, { '$' }
        %00000000, { '%' }
        %00000000, { '&' }
        %00000000, { ''' }
        %00111001, { '(' }
        %00001111, { ')' }
        %00000000, { '*' }
        %00000000, { '+' }
        %00000000, { ',' }
        %01000000, { '-' }
        %00000000, { '.' }
        %01001001, { '/' }
        %00111111, { '0' }
        %00000110, { '1' }
        %01011011, { '2' }
        %01001111, { '3' }
        %01100110, { '4' }
        %01101101, { '5' }
        %01111101, { '6' }
        %00100111, { '7' }
        %01111111, { '8' }
        %01101111, { '9' }
        %00000000, { ':' }
        %00000000, { ';' }
        %00000000, { '<' }
        %01001000, { '=' }
        %00000000, { '>' }
        %00000000, { '?' }
        %00000000, { '@' }
        %01110111, { 'A' }
        %01111100, { 'B' }
        %00111001, { 'C' }
        %01011110, { 'D' }
        %01111001, { 'E' }
        %01110001, { 'F' }
        %00000000, { 'G' }
        %01110110, { 'H' }
        %00000000, { 'I' }
        %00000000, { 'J' }
        %00000000, { 'K' }
        %00000000, { 'L' }
        %00000000, { 'M' }
        %01010100, { 'N' }
        %01011100, { 'O' }
        %01110011, { 'P' }
        %01010111, { 'Q' }
        %01010000, { 'R' }
        %01101101, { 'S' }
        %01111000, { 'T' }
        %00011100);{ 'U' }

      HexChar: Array [0..15] of Char = '0123456789ABCDEF';

      BlockStart = $AA;
      BlockStartLNG = $CA;

      bpBlockStart         = 0;
      bpDestinationAddress = 1;
      bpcommand            = 2;
      bpParameter1         = 3;
      bpParameter2         = 4;
      bpSourceAddress      = 5;
      bpChecksum           = 6;

      bpParameter3         = 7;
      bpParameter4         = 8;
      bpParameter5         = 9;
      bpParameter6         = 10;
      bpParameter7         = 11;
      bpParameter8         = 12;
      bpParameter9         = 13;
      bpParameter10        = 14;
      bpParameter11        = 15;
      bpParameter12        = 16;

      LNG_ModuleID         = $A0;

      Cmd_Relay                 = $21;
      Cmd_RelayTime             = $22;
      Cmd_GetData               = $31;
      Cmd_StatusRequest         = $40;
      Cmd_Status                = $41;
      Cmd_SoftwareVersion       = $4F;
      Cmd_ActivateOutput        = $54;
      Cmd_ChangeAddress         = $55;
      Cmd_LNG_Tx                = $57;
      Cmd_LearnLNG_Tx           = $58;
      Cmd_Cfg                   = $59;

Type  Bits64 = Array [0..7] of Byte;
      Bits32 = Array [0..3] of Byte;

      TXKeyData = Record
                    DecriptionKey: Bits64;
                    SerialNum: Bits32;
                    KeyCounter: Word;
                  end;

      TSetup = (None, LearnKey, LearnKey2, SetModuleType, SetChannel, SetChannelType, SetTxKey, SetBlindsMode, SetTime);

      TBlindsMode = (bmNone, bmCh12, bmCh34, bmCh1234);

      TKbd = (kNone, kEnter, kCancel, kUp, kDown);

      TChType = (chFollow, chToggle, chPulse);

      TModuleType = (mtNormal, mtCrawford, mtDoor, mtAlarm);

      TModuleFlags = (mfBlinds12, mfBlinds34);

      TBlindsState = (bsIdle, bsOut1, bsIdleBetween, bsOut2);

      TModuleFlagsSet = Set of TModuleFlags;

      TDoorState = (dsStop, dsUp, dsDown);

Const KeyTable: Array [kNone..kDown] of Byte = (
        $0F,
        $07,
        $0B,
        $0D,
        $0E);

Var   RS485_TX:         Boolean absolute P3.3; Volatile;
      I2C.SCL:          Boolean absolute P3.4; Volatile;
      I2C.SDA:          Boolean absolute P3.5; Volatile;
      Strobe:           Boolean absolute P3.6; Volatile;
      Key_Down:         Boolean absolute P2.0; Volatile;
      Key_Up:           Boolean absolute P2.1; Volatile;
      Key_Cancel:       Boolean absolute P2.2; Volatile;
      Key_Enter:        Boolean absolute P2.3; Volatile;
      IN5:              Boolean absolute P2.5; Volatile;
      IN6:              Boolean absolute P2.7; Volatile;
      Aux_Strobe:       Boolean absolute P0.2; Volatile;
      Aux_SDA:          Boolean absolute P0.1; Volatile;
      Aux_SCL:          Boolean absolute P0.0; Volatile;
      IN1:              Boolean absolute P1.0; Volatile;
      IN2:              Boolean absolute P1.1; Volatile;
      IN3:              Boolean absolute P1.2; Volatile;
      IN4:              Boolean absolute P1.3; Volatile;
      OUT1:             Boolean absolute P1.6; Volatile;
      OUT2:             Boolean absolute P1.4; Volatile;
      OUT3:             Boolean absolute P1.7; Volatile;
      OUT4:             Boolean absolute P1.5; Volatile;
      RF_Data:          Boolean absolute P3.2; Volatile;
      AuxModule:        Boolean absolute P2.4; Volatile;
      WatchDogReset:    Boolean absolute SCL;  Volatile;
      AutoSaveDisabled: Boolean absolute IN5;
      CounterCheck:     Boolean absolute IN6;

      T2CON:    Byte absolute $C8; Volatile;
      T2MOD:    Byte absolute $C9; Volatile;
      RCAP2L:   Byte absolute $CA; Volatile;
      RCAP2H:   Byte absolute $CB; Volatile;
      TL2:      Byte absolute $CC; Volatile;
      TH2:      Byte absolute $CD; Volatile;

    { T2CON }
      TF2:      Boolean absolute T2CON.7; Volatile;
      EXF2:     Boolean absolute T2CON.6; Volatile;
      RCLK:     Boolean absolute T2CON.5; Volatile;
      TCLK:     Boolean absolute T2CON.4; Volatile;
      EXEN2:    Boolean absolute T2CON.3; Volatile;
      TR2:      Boolean absolute T2CON.2; Volatile;
      C_T2:     Boolean absolute T2CON.1; Volatile;
      CP_RL2:   Boolean absolute T2CON.0; Volatile;

    { IE}
      ET2:      Boolean absolute IE.5; Volatile;

    { IP }
      PT2:      Boolean absolute IP.5; Volatile;

    { P1 }
      T2EX:     Boolean absolute P1.1; Volatile;
      T2:       Boolean absolute P1.0; Volatile;

      NewBit,
      KeyPacketReceived:          Boolean;
      Timer:                      Word_; Volatile;
      TempWord:                   Word_;
      TempCounter:                Word_;
      TempByte:                   Byte;
      BitCounter:                 Byte;
      TE:                         Word_ IDATA;
      CounterDifference:          Word_ absolute TE;
      EncryptedCode:              Bits32 IDATA;
      SerialNumber:               Bits32 IDATA;
      DataByte:                   Bits64 absolute EncryptedCode;
      Counter:                    Word_  absolute EncryptedCode;
      Key:                        Bits64 IDATA;
      CalculatedKey:              Bits64 IDATA;
      I:                          Byte;
      KeyProcessingTimer:         Word; Volatile;
      Vlow, RPT, DecryptOK:       Boolean;
      PreambleTimer:              Word; Volatile;
      Threshold, EquilizingTime:  Word IDATA;
      NumberOfLearnedKeys:        Word IDATA;
      BlinkTimer:                 Word_; Volatile;
      TempBits32:                 Bits32 IDATA;
      Blinds12State, Blinds34State: TBlindsState;

      Display:                    Char;
      RF_led:                     Boolean;
      RX_Led:                     Boolean;
      TX_Led:                     Boolean;
      Sending,
      BlockReceived:              Boolean;
      RX_Counter,
      TX_Counter,
      RX_Checksum:                Byte;
      RX_Buffer,
      TX_Buffer:                  Array [0..16] of Byte IDATA;
      RS485_Timer,
      RX_LedTimer,
      TX_LedTimer:                Byte; Volatile;
      ModuleAddress:              Byte;
      DelayTimer:                 Word IDATA; Volatile;
      Relay:                      Array [1..15] of Byte;
      RelayTimer:                 Array [1..4] of Word; Volatile;
      RelayTime:                  Array [1..4] of Word_ IDATA;
      RelayTimeBlock:             Array [0..7] of Byte absolute RelayTime;

      Setup:                      TSetup;
      BlindsMode, TempBlindsMode: TBlindsMode;
      SetupTimer:                 Word; Volatile;
      TempSet:                    Boolean;
      TempTime:                   Byte;
      Kbd:                        TKbd;
      KeyBits,
      LastKeyBits:                Byte;
      ModuleType:                 TModuleType;
      ModuleFlags:                TModuleFlagsSet IDATA;
      ChData:                     Array [1..15] of Byte IDATA;
      ChDataBlock:                Array [0..14] of Byte absolute ChData;
      TxKeyBits,
      LastTxKeyBits:              Byte;
      ClsTimer:                   Word IDATA; Volatile;
      LastInput1,
      LastInput2,
      LastInput3,
      LastInput4:                 Boolean;
      Input1,
      Input2,
      Input3,
      Input4:                     Boolean;
      DoorState:                  TDoorState;
      LastDoorState:              TDoorState IDATA;
      TempModuleType:             TModuleType;
      Channel:                    Byte;
      TempChannelType:            TChType;
      BlockSize:                  Byte;
      NoKeyTimer:                 Word; Volatile;
      Latch:                      Word_ IDATA;
      Mask:                       Byte IDATA;
      Relay5_15Timer:             Byte IDATA; Volatile;
      Relay5_15:                  Byte IDATA;
      LowThreshold,
      HighThreshold:              Word IDATA;
      TXPositionInEEprom:         Word IDATA;

Procedure TimerProc; Interrupt Timer0; Using 2; { 1 ms interrupt }
begin
  TL0 :=  Lo (Const1ms);
  TH0 :=  Hi (Const1ms);

  Inc (BlinkTimer.W);

  Inc (KeyProcessingTimer);
  If DelayTimer <> 0 then Dec (DelayTimer);
  If RS485_Timer <> 0 then Dec (RS485_Timer) else RS485_TX := False;
  If RX_LedTimer <> 0 then Dec (RX_LedTimer) else RX_Led := False;
  If TX_LedTimer <> 0 then Dec (TX_LedTimer) else TX_Led := False;

  If NoKeyTimer <> 0 then Dec (NoKeyTimer);

  If SetupTimer <> 0 then
    begin
      Dec (SetupTimer);
      If SetupTimer = 0 then Setup := None;
    end;

  If ClsTimer <> 0 then Dec (ClsTimer);

  If BlinkTimer.L and $1F = 0 then
    begin
      If Relay5_15Timer <> 0 then
        begin
          Dec (Relay5_15Timer);
          If Relay5_15Timer = 0 then
            Relay [Relay5_15] := Byte (Relay_OFF);
        end;

      If RelayTimer [1] <> 0 then
        begin
          Dec (RelayTimer [1]);
          If RelayTimer [1] = 0 then
            begin
              Relay [1] := Byte (Relay_OFF);
              OUT1 := Relay_OFF;
              If Blinds12State = bsOut1 then Blinds12State := bsIdleBetween;
            end;
        end;

      If RelayTimer [2] <> 0 then
        begin
          Dec (RelayTimer [2]);
          If RelayTimer [2] = 0 then
            begin
              Relay [2] := Byte (Relay_OFF);
              OUT2 := Relay_OFF;
              If Blinds12State = bsOut2 then Blinds12State := bsIdle;
            end;
        end;

      If RelayTimer [3] <> 0 then
        begin
          Dec (RelayTimer [3]);
          If RelayTimer [3] = 0 then
            begin
              Relay [3] := Byte (Relay_OFF);
              OUT3 := Relay_OFF;
              If Blinds34State = bsOut1 then Blinds34State := bsIdleBetween;
            end;
        end;

      If RelayTimer [4] <> 0 then
        begin
          Dec (RelayTimer [4]);
          If RelayTimer [4] = 0 then
            begin
              Relay [4] := Byte (Relay_OFF);
              OUT4 := Relay_OFF;
              If Blinds34State = bsOut2 then Blinds34State := bsIdle;
            end;
        end;
    end;
end;

Procedure Timer1Proc; Interrupt Timer1; { Time after pulse received, 8 ms interrupt if no RF data pulse }
begin
  TL1 :=  Lo (TimerConst);
  TH1 :=  Hi (TimerConst);
  RF_Led := False;
  BitCounter := 0;
  PreambleTimer := 0;
end;

Procedure Int0Proc; Interrupt External0; Using 1; { RF Data Input }
begin
  TR1 := False;
  Timer.L := TL1;
  Timer.H := TH1;
  TL1 :=  Lo (TimerConst);
  TH1 :=  Hi (TimerConst);
  TR1 := True;
  NewBit := True;
end;

Procedure SerialProc; Interrupt Serial; Using 3; { RS485 }
Var Ch: Byte;
begin
  If TI then
    begin
      TI := False;
      If TX_Counter < BlockSize then
        begin
          TX_LedTimer := LedTime;
          RS485_Timer := RS485_Time;
          SBUF := TX_Buffer [TX_Counter];
          Inc (TX_Counter);
        end else Sending := False;

    end;

  If RI then
    begin
      RI := False;
      If Sending or BlockReceived then Exit;
      RX_LedTimer := LedTime;
      RX_Led := True;
      If ClsTimer <> 0 then Exit;
      Ch := SBUF;
      If (RX_Counter = bpBlockStart) then
        Case Ch of
          BlockStart: BlockSize := 7;
          BlockStartLNG: BlockSize := 17;
          else Exit;
        end;
      RX_Buffer [RX_Counter] := Ch;
      Inc (RX_Counter);
      RX_Checksum := RX_Checksum xor Ch;
      If RX_Counter = BlockSize then
        begin
          If ((RX_Buffer [bpDestinationAddress] = $FF) or (RX_Buffer [bpDestinationAddress] = ModuleAddress))
            and (RX_Checksum = 0) then BlockReceived := True;
          RX_Counter := 0;
          RX_Checksum := 0;
        end;
    end;
end;

Function HighNib (Value: Byte): Byte;
begin
  HighNib := Value shr 4;
end;

Function LowNib (Value: Byte): Byte;
begin
  LowNib := Value and $0F;
end;

Function NibToByte (Hi, Lo: Byte): Byte;
begin
  NibToByte := Hi shl 4 or Lo and $0F;
end;

Function ManufacturerKey (Pos: Byte): Byte;
begin
  ManufacturerKey := Byte (ManufacturerKeyData [Pos xor $07]);
end;

Procedure RestartWatchDog;
begin
  WatchDogReset := LowLevel;
  Asm
    NOP
    NOP
  end;
  WatchDogReset := HighLevel;
end;

Procedure WriteDisplay;
Var Nw: Byte;
begin
  Latch.W := Segments [Display];
  If Setup = None then
    begin
      If (BlinkTimer.H and $2) = 0 then Latch.L := Latch.L or $80;
    end else If (BlinkTimer.L and $80) = 0 then Latch.L := Latch.L or $80;
  If RF_Led then Latch.H := Latch.H or $04;
  If RX_Led then Latch.H := Latch.H or $02;
  If TX_Led then Latch.H := Latch.H or $01;
  Latch.H := Latch.H xor $FF;
  Latch.L := Latch.L xor $FF;

  SCL := LowLevel;
  SCL := HighLevel;
  Asm
    NOP
  end;
  SCL := LowLevel;
  SDA := False;

  For Nw := 1 to 16 do
    begin
      SDA := (Latch.H and $80) <> 0;
      SCL := HighLevel;
      Asm
        NOP
      end;
      SCL := LowLevel;
      Latch.W := Latch.W + Latch.W;
    end;
  Strobe := HighLevel;
  Asm
    NOP
  end;
  Strobe := LowLevel;
  SCL := HighLevel;
end;

Procedure Delay (Dl: Word);
begin
  DelayTimer := Dl;
  Repeat
    WriteDisplay;
  until DelayTimer = 0;
end;

Procedure Send;
begin
  TX_Counter := 0;
  RS485_Timer := RS485_Time;
  RS485_TX := True;
  TX_LedTimer := LedTime;
  TX_Led := True;
  Repeat
    WriteDisplay;
  until RS485_Timer <= 1;
  TI := True;
end;

Procedure SendBlock;
Var M, K: Byte;
begin
  While Sending do WriteDisplay;
  Sending := True;
  TX_Buffer [bpSourceAddress] := ModuleAddress;
  TX_Buffer [bpBlockStart] := BlockStart;
  M := 0;
  For K := 0 to 5 do M := M xor TX_Buffer [K];
  TX_Buffer [bpChecksum] := M;
  BlockSize := 7;
  Send;
end;

Procedure SendLNGBlock;
Var M, K: Byte;
begin
  While Sending do WriteDisplay;
  Sending := True;
  TX_Buffer [bpSourceAddress] := ModuleAddress;
  TX_Buffer [bpBlockStart] := BlockStartLNG;
  TX_Buffer [bpChecksum] := 0;
  M := 0;
  For K := 0 to 16 do M := M xor TX_Buffer [K];
  TX_Buffer [bpChecksum] := M;
  BlockSize := 17;
  Send;
end;

Procedure SendToAuxModule;
Var Nw: Byte;
begin
  Aux_SCL := LowLevel;
  Asm
    NOP
    NOP
    NOP
    NOP
  end;
  Aux_SCL := HighLevel;
  Asm
    NOP
    NOP
    NOP
    NOP
  end;

  For Nw := 5 to 15 do
    begin
      Aux_SDA := Relay [Nw] = Byte (Relay_ON);
      Asm
        NOP
        NOP
        NOP
        NOP
      end;
      WatchDogReset := LowLevel;
      Asm
        NOP
      end;
      WatchDogReset := HighLevel;
      Aux_SCL := LowLevel;
      Asm
        NOP
        NOP
        NOP
        NOP
      end;
      Aux_SCL := HighLevel;
    end;

  Asm
    NOP
    NOP
    NOP
    NOP
  end;
  Aux_Strobe := LowLevel;
  Asm
    NOP
    NOP
    NOP
    NOP
  end;
  Aux_Strobe := HighLevel;
end;

Procedure SendStatus;
begin
  TX_Buffer [bpCommand] := Cmd_Status;
  If (Setup = LearnKey) or (Setup = LearnKey2) then
    TX_Buffer [bpParameter2] := LNG_ModuleID or $01 else
      TX_Buffer [bpParameter2] := LNG_ModuleID;
  If Hi (NumberOfLearnedKeys) = 0 then
    TX_Buffer [bpParameter1] := Lo (NumberOfLearnedKeys) else
      TX_Buffer [bpParameter1] := 255;
  SendBlock;
end;

Procedure SendToTerminal;
Var LRC: Byte;

  Procedure SendByteToTerminal (BCR: Byte);
  Var TempByte: Byte;
  begin
    LRC := LRC xor BCR;
    If BCR < $80 then
      begin
        BCR := BCR and $0F;
        Asm
          MOV   A, BCR;
        end;
        If not P then BCR := BCR or $10;
      end;
    For TempByte := 1 to 5 do
      begin
        Aux_SDA := BCR and $01 <> $00;
        Delay (1);
        Aux_SCL := HighLevel;
        Delay (1);
        Aux_SCL := LowLevel;
        Delay (1);
        BCR := BCR shr 1;
      end;
  end;

  Procedure SendManufacturerKey;
  begin
    SendByteToTerminal (HighNib (I));
    SendByteToTerminal (LowNib  (I));
  end;

begin
  Aux_Strobe := HighLevel;
  Delay (1);
  LRC := 0;

  SendByteToTerminal ($80);
  SendByteToTerminal ($80);

  SendByteToTerminal ($B);

  I := ManufacturerKey (7);
  SendManufacturerKey;
  I := ManufacturerKey (6);
  SendManufacturerKey;
  I := ManufacturerKey (5);
  SendManufacturerKey;

  SendByteToTerminal ($D);
  SendByteToTerminal (HighNib (SerialNumber [2]));
  SendByteToTerminal (LowNib  (SerialNumber [2]));
  SendByteToTerminal (HighNib (SerialNumber [1]));
  SendByteToTerminal (LowNib  (SerialNumber [1]));
  SendByteToTerminal (HighNib (SerialNumber [0]));
  SendByteToTerminal (LowNib  (SerialNumber [0]));
  SendByteToTerminal ($D);
  SendByteToTerminal (TxKeyBits div 10);
  SendByteToTerminal (TxKeyBits mod 10);

  SendByteToTerminal ($D);
  SendByteToTerminal (Byte (not Vlow));

  SendByteToTerminal ($F);
  SendByteToTerminal (LRC and $0F);

  Delay (1);
  Aux_Strobe := LowLevel;
  Aux_SDA := LowLevel;
end;

Function Check_EEPROM_Signature: Boolean;
begin
  Check_EEPROM_Signature := False;
  If ReadByteFromEEPROM (Ofs (TEEprom.EE_Signature) + 0) <> Lo (Signature) then Exit;
  If ReadByteFromEEPROM (Ofs (TEEprom.EE_Signature) + 1) <> Hi (Signature) then Exit;
  Check_EEPROM_Signature := True;
end;

Procedure WriteRelayTimesToEEPROM;
Var TempByte: Byte;
begin
  For TempByte := 0 to 7 do
    WriteByteToEEPROM (Ofs (TEEprom.EE_RelayTime [1]) + TempByte, RelayTimeBlock [TempByte]);
end;

Procedure InitNormalModuleType;
Var TempByte: Byte;
begin
  ModuleType := mtNormal;
  ModuleFlags := [];
  For TempByte := 1 to 15 do ChData [TempByte] := TempByte shl 4;
  RelayTime [1].W := DafaultRelayLongPulseTime;
  RelayTime [2].W := DafaultRelayLongPulseTime;
  RelayTime [3].W := DafaultRelayLongPulseTime;
  RelayTime [4].W := DafaultRelayLongPulseTime;
  WriteRelayTimesToEEPROM;
end;

Procedure WriteModuleConfig;
Var TempByte: Byte;
begin
  WriteByteToEEPROM (Ofs (TEEprom.EE_ModuleType), Byte (ModuleType));
  WriteByteToEEPROM (Ofs (TEEprom.EE_ModuleFlags), Byte (ModuleFlags));
  For TempByte := 0 to 14 do
    WriteByteToEEPROM (Ofs (TEEprom.EE_ChannelConfig) + TempByte, ChDataBlock [TempByte]);
end;

Procedure Init_EEPROM;
begin
  WriteByteToEEPROM (Ofs (TEEprom.EE_Signature) + 0, Lo (Signature));
  WriteByteToEEPROM (Ofs (TEEprom.EE_Signature) + 1, Hi (Signature));
  NumberOfLearnedKeys := 0;
  WriteByteToEEPROM (Ofs (TEEprom.EE_NumberOfLearnedKeysLo), 0);
  WriteByteToEEPROM (Ofs (TEEprom.EE_NumberOfLearnedKeysHi), 0);
  ModuleAddress := $90;
  WriteByteToEEPROM (Ofs (TEEprom.EE_ModuleAddress), ModuleAddress);
  InitNormalModuleType;
  WriteModuleConfig;
end;

Procedure Read_EEPROM;
Var TempByte: Byte;
begin
  Lo (NumberOfLearnedKeys) := ReadByteFromEEPROM (Ofs (TEEprom.EE_NumberOfLearnedKeysLo));
  Hi (NumberOfLearnedKeys) := ReadByteFromEEPROM (Ofs (TEEprom.EE_NumberOfLearnedKeysHi));

  ModuleAddress := ReadByteFromEEPROM (Ofs (TEEprom.EE_ModuleAddress));
  For TempByte := 0 to 7 do
    RelayTimeBlock [TempByte] := ReadByteFromEEPROM (Ofs (TEEprom.EE_RelayTime [1]) + TempByte);

  ModuleType := TModuleType (ReadByteFromEEPROM (Ofs (TEEprom.EE_ModuleType)));
  Byte (ModuleFlags) := ReadByteFromEEPROM (Ofs (TEEprom.EE_ModuleFlags));
  For TempByte := 0 to 14 do
    ChDataBlock [TempByte] := ReadByteFromEEPROM (Ofs (TEEprom.EE_ChannelConfig) + TempByte);
end;

Procedure Init;
Var K: Byte;
    TempByte: Byte;
begin
  RS485_TX := LowLevel;
  RF_Led := False;
  RX_Led := False;
  TX_Led := False;
  For TempByte := 1 to 15 do
    Relay [TempByte] := Byte (Relay_OFF);
  Strobe := LowLevel;
  Aux_SDA := LowLevel;        { Inverted output is HIGH to comply with ISO Track 2 standard }
  Aux_SCL := LowLevel;
  Aux_Strobe := AuxModule;
  If AuxModule then
    begin
      Mask := $0F
    end else
      begin
        Mask := $03;
      end;

  LastKeyBits := $0F;
  RelayTimer [1] := 0;
  RelayTimer [2] := 0;
  RelayTimer [3] := 0;
  RelayTimer [3] := 0;
  Relay5_15Timer := 0;

  Sending := False;
  BlockReceived := False;
  RX_Counter := 0;
  TX_Counter := 0;
  RX_Checksum := 0;
  Setup := None;
  SetupTimer := 0;
  Blinds12State  := bsIdle;
  Blinds34State  := bsIdle;
  ClsTimer := 0;
  DoorState := dsStop;
  LastDoorState := dsUp;

  NewBit := False;
  KeyPacketReceived := False;
  BitCounter := 0;
  PreambleTimer := 0;

  TL0 :=  Lo (Const1ms);
  TH0 :=  Hi (Const1ms);
  TL1 :=  Lo (TimerConst);
  TH1 :=  Hi (TimerConst);

  RCAP2L :=  Lo (BaudRateTimerValue);
  RCAP2H :=  Hi (BaudRateTimerValue);
  TMOD := %00010001; { Timer1: no GATE, Timer, 16 bit timer }
                     { Timer0: no GATE, Timer, 16 bit timer }
  TCON := %01010101; { Timer 1 run, Timer 0 run }
                     { Int1 falling edge, Int0 falling edge }
  T2CON := %00110100;{ Baud rate generator, Run }
  IE    := %10011011;{ Serial, Timer1, Timer0, Int0 }
  PX0 := True;       { External Interrupt 0 Priority }

  SCON := %01010000; { Serial Mode 1, Enable Reception }

  Display := ' ';
  WriteDisplay;
  If AuxModule then SendToAuxModule;
  Delay (100);

  If Key_Cancel = LowLevel then Init_EEPROM
    else begin
           If Check_EEPROM_Signature then Read_EEPROM
             else begin
                    Delay (100);
                    If Check_EEPROM_Signature then Read_EEPROM
                      else Init_EEPROM;
                  end;
         end;
  Delay (100);

  If not Check_EEPROM_Signature then
    Repeat
      Display := 'E';
      WriteDisplay;
    until False;

  Delay (200);

  K := 0;
  Repeat
    TempByte := Byte (GreetingString [K]);
    Inc (K);
    If TempByte <> 0 then
      begin
        Display := Char (TempByte);
        Delay (300);
        Display := ' ';
        Delay (100);
      end;
  until TempByte = 0;
  Delay (500);
  Display := HexChar [LowNib (VersionHi)];
  Delay (300);
  Display := ' ';
  Delay (100);
  Display := HexChar [HighNib (VersionLo)];
  Delay (300);
  Display := ' ';
  Delay (100);
  Display := HexChar [LowNib (VersionLo)];
  Delay (300);
  Display := ' ';
end;

Procedure SetManufacturerKey;
Var I: Byte;
begin
  For I := 0 to 7 do Key [I] := ManufacturerKey (I);
end;

Procedure SetCalculatedKey;
Var I: Byte;
begin
  For I := 0 to 7 do Key [I] := CalculatedKey [I];
end;

Function CalcBit: Boolean;
Var PulseDuraton: Word;
begin
  PulseDuraton := Timer.W - TimerConst - EquilizingTime;
  If (PulseDuraton < LowThreshold) or (PulseDuraton > HighThreshold) then
    begin
      RF_Led := False;
      BitCounter := 0;
      PreambleTimer := 0;
      Exit;
    end;
  RF_Led := True;
  Result := PulseDuraton < Threshold;
  If Result then EquilizingTime := TE.W else EquilizingTime := 0;
end;

Procedure SetBit (B0: Byte);
Var ByteCounter: Byte;
begin
  ByteCounter := B0 shr 3;
  DataByte [ByteCounter] := DataByte [ByteCounter] shr 1;
  If CalcBit then DataByte [ByteCounter] := DataByte [ByteCounter] or $80;
end;

Procedure StoreBit;
begin
  Inc (BitCounter);
  Case BitCounter of
    5..12: PreambleTimer := PreambleTimer + (Timer.W - TimerConst);
    13: begin
          TE.H := HighNib (Hi (PreambleTimer));
          TE.L := NibToByte (LowNib (Hi (PreambleTimer)), HighNib (Lo (PreambleTimer)));   { Divide by 16 }
          LowThreshold  := TE.W + TE.W div 2;
          HighThreshold := TE.W * 3 + TE.W div 2;
          Threshold := TE.W + TE.W + TE.W shr 1;
          EquilizingTime := TE.W + TE.W;
          EquilizingTime := EquilizingTime + EquilizingTime;
          EquilizingTime := EquilizingTime + EquilizingTime;
          EquilizingTime := EquilizingTime +  TE.W;
          SetBit (0);
          DataByte [7] := 0;
        end;
    14..76: SetBit (BitCounter - 13);
    77: begin
          Vlow := CalcBit;
       end;
    78: begin
          RPT := CalcBit;
          BitCounter := 0;
          PreambleTimer := 0;
          KeyPacketReceived := True;
          ET0 := False;
          KeyProcessingTimer := 0;
          ET0 := True;
        end;
  end;
  NewBit := False;
end;

Procedure SetRelay (R, Status: Byte);
Var BoolSt: Boolean;
begin
  Case Status of
    0: begin
         BoolSt := Relay_OFF;
         Case R of
           1..4: RelayTimer [R] := 0;
           else Relay5_15Timer := 0;
         end;
       end;
    1: BoolSt := Relay_ON;
    2: BoolSt := not Boolean (Relay [R]);
  end;
  Relay [R] := Byte (BoolSt);
  Case R of
    1: OUT1 := BoolSt;
    2: OUT2 := BoolSt;
    3: OUT3 := BoolSt;
    4: OUT4 := BoolSt;
  end;
end;

Procedure  Decrypt;
Const Table: Array [0..31] of Boolean = (
        False,
        True,
        True,
        True,
        False,
        True,
        False,
        False,
        False,
        False,
        True,
        False,
        True,
        True,
        True,
        False,
        False,
        False,
        True,
        True,
        True,
        False,
        True,
        False,
        False,
        True,
        False,
        True,
        True,
        True,
        False,
        False);

Var Count: Word;
    Index: Byte;
    Carry: Boolean;
begin
  DecryptOK := False;
  For Count := 0 to 528 - 1 do
    begin
      WatchDogReset := LowLevel;
      Asm
        NOP
      end;
      WatchDogReset := HighLevel;
      Index := 0;

      If EncryptedCode [0] and $01 <> 0 then Index := Index or $01;
      If EncryptedCode [1] and $01 <> 0 then Index := Index or $02;
      If EncryptedCode [2] and $08 <> 0 then Index := Index or $04;
      If EncryptedCode [3] and $02 <> 0 then Index := Index or $08;
      If EncryptedCode [3] and $40 <> 0 then Index := Index or $10;

      Carry := (EncryptedCode [1] and $80 <> 0) xor
               (EncryptedCode [3] and $80 <> 0) xor
               (Key [1] and $80 <> 0) xor Table [Index];

      Asm
        MOV    C, Carry
        MOV    R0, #EncryptedCode
        MOV    B, #4
@LOOP_1:
        MOV    A, @R0
        RLC    A
        MOV    @R0, A
        INC    R0
        DJNZ   B, @LOOP_1

        MOV    R0, #Key + 7
        MOV    A, @R0
        RLC    A
        MOV    R0, #Key + 0
        MOV    B, #8
@LOOP_2:
        MOV    A, @R0
        RLC    A
        MOV    @R0, A
        INC    R0
        DJNZ   B, @LOOP_2
      end;
    end;
  If EncryptedCode [3] and $F0 <> SerialNumber [3] and $F0 then Exit;       { Check if both key sets are equal }
  DecryptOK := True;
end;

Procedure GenerateKey;
begin
  EncryptedCode [0] := SerialNumber [0];
  EncryptedCode [1] := SerialNumber [1];
  EncryptedCode [2] := SerialNumber [2];
  EncryptedCode [3] := (SerialNumber [3] and $0F) or $20;
  SetManufacturerKey;
  Decrypt;
  CalculatedKey [0] := EncryptedCode [0];
  CalculatedKey [1] := EncryptedCode [1];
  CalculatedKey [2] := EncryptedCode [2];
  CalculatedKey [3] := EncryptedCode [3];

  EncryptedCode [0] := SerialNumber [0];
  EncryptedCode [1] := SerialNumber [1];
  EncryptedCode [2] := SerialNumber [2];
  EncryptedCode [3] := (SerialNumber [3] and $0F) or $60;
  SetManufacturerKey;
  Decrypt;
  CalculatedKey [4] := EncryptedCode [0];
  CalculatedKey [5] := EncryptedCode [1];
  CalculatedKey [6] := EncryptedCode [2];
  CalculatedKey [7] := EncryptedCode [3];
end;

Function TXInEEprom: Boolean;
Var I: Word;
Label NoMatch;
begin
  TXInEEprom := False;
  TempWord.W := Ofs (TEEprom.EE_TXKeyData);
  I := 1;
  While I <= NumberOfLearnedKeys do
    begin
      Asm
        MOV       A, TempWord
        ADD       A, #LOW  (8)
        MOV       DPL, A
        MOV       A, TempWord+1
        ADDC      A, #HIGH (8)
        MOV       DPH, A
        LCALL     ReadBytesFromEEPROM
        MOV       R0, #SerialNumber
        MOV       A, R2
        XRL       A, @R0
        JNZ       NoMatch
        INC       R0
        MOV       A, R3
        XRL       A, @R0
        JNZ       NoMatch
        INC       R0
        MOV       A, R4
        XRL       A, @R0
        JNZ       NoMatch
        INC       R0
        MOV       A, R5
        XRL       A, @R0
        ANL       A, #$0F
        JNZ       NoMatch
      end;

      TXInEEprom := True;
      TXPositionInEEprom := I;
      Break;

NoMatch:

      Inc (TempWord.W, SizeOf (TxData));
      Inc (I);
    end;
end;

Function SaveTxToEEprom: Boolean;
Var I: Byte;
begin
  SaveTxToEEprom := True;
  If not TXInEEprom then
    begin
      TempWord.W := Ofs (TEEprom.EE_TXKeyData) + SizeOf (TxData) * NumberOfLearnedKeys;

     { Try to write to the EEPROM at the last location for new key }

      WriteByteToEEPROM (TempWord.W + SizeOf (TxData) - 1, $55);
      SaveTxToEEprom := ReadByteFromEEprom (TempWord.W + SizeOf (TxData) - 1) = $55;
      Delay (3);
      If Result then
        begin
          WriteByteToEEPROM (TempWord.W + SizeOf (TxData) - 1, $AA);
          SaveTxToEEprom := ReadByteFromEEprom (TempWord.W + SizeOf (TxData) - 1) = $AA;
          Delay (3);
        end;

      If not Result then
        begin
          Display := 'F';
          Delay (500);
          Exit;
        end;

      Inc (NumberOfLearnedKeys);
    end;
  For I := 0 to 7 do WriteByteToEEPROM (TempWord.W + Ofs (TxData.tdKey)              + I, CalculatedKey [I]);
  For I := 0 to 2 do WriteByteToEEPROM (TempWord.W + Ofs (TxData.tdSerialNumber [0]) + I, SerialNumber [I]);
  WriteByteToEEPROM (TempWord.W + Ofs (TxData.tdSerialNumber [3]), SerialNumber [3] and $0F);
  WriteByteToEEPROM (TempWord.W + Ofs (TxData.tdCounter.L), Counter.L);
  WriteByteToEEPROM (TempWord.W + Ofs (TxData.tdCounter.H), Counter.H);

  WriteByteToEEPROM (Ofs (TEEprom.EE_NumberOfLearnedKeysLo), Lo (NumberOfLearnedKeys));
  WriteByteToEEPROM (Ofs (TEEprom.EE_NumberOfLearnedKeysHi), Hi (NumberOfLearnedKeys));
end;

Procedure RemoveTx;
Var LastTxAddress: Word;
    I: Byte;
begin
  If not TXInEEprom then Exit;
  Display := 'D';
  WriteDisplay;
  If NumberOfLearnedKeys > 1 then If TXPositionInEEprom <> NumberOfLearnedKeys then
    begin
      LastTxAddress := Ofs (TEEprom.EE_TXKeyData) + SizeOf (TxData) * Word (NumberOfLearnedKeys - 1);
      For I := 0 to 13 do
        WriteByteToEEPROM (TempWord.W + I, ReadByteFromEEPROM (LastTxAddress + I));
    end;
  Dec (NumberOfLearnedKeys);
  WriteByteToEEPROM (Ofs (TEEprom.EE_NumberOfLearnedKeysLo), Lo (NumberOfLearnedKeys));
  WriteByteToEEPROM (Ofs (TEEprom.EE_NumberOfLearnedKeysHi), Hi (NumberOfLearnedKeys));
  Delay (20);
  Repeat
    Delay (1);
  until not Sending;
  Delay (2);
  TX_Buffer [bpDestinationAddress] := $FF;
  SendStatus;
  Delay (200);
end;

Procedure DelayRelayTime1;
begin
  Delay (32 * RelayTime [1].W);
end;

Procedure DelayRelayTime2;
begin
  Delay (32 * RelayTime [2].W);
end;

Procedure DoorMotorOff;
begin
  SetRelay (2, 0); { Motor Off }
  DelayRelayTime1;
  SetRelay (3, 0); { Lock Off }
end;

Procedure LightOnAndDelay;
begin
  If Relay [4] = Byte (Relay_ON) then Exit;
  RelayTimer [4] := RelayTime [4].W;
  SetRelay (4, 1);
  DelayRelayTime2;
end;

Procedure ProcessDoor;
begin
  Case DoorState of
    dsStop: begin
              If IN4 = LowLevel then
                If IN2 = LowLevel then { Open Switch and LightCell = OK }
                  begin
                    Case ModuleType of
                      mtCrawford: begin
                                    LightOnAndDelay;
                                    RelayTimer [3] := RelayTime [1].W; { Down }
                                    SetRelay (3, 1);
                                  end;
                      mtDoor: begin
                                SetRelay (1, MotorDown); { Direction Down }
                                SetRelay (3, 1); { Lock Off }
                                DelayRelayTime1;
                                SetRelay (2, 1); { Motor On }
                              end;
                    end;
                    DoorState := dsDown;
                    LastDoorState := dsDown;
                  end;
              If IN3 = LowLevel then { Closed Switch }
                begin
                  Case ModuleType of
                    mtCrawford: begin
                                  LightOnAndDelay;
                                  RelayTimer [1] := RelayTime [1].W; { Up }
                                  SetRelay (1, 1);
                                end;
                    mtDoor: begin
                              SetRelay (1, MotorUp); { Direction Up }
                              SetRelay (3, 1); { Lock Off }
                              DelayRelayTime1;
                              SetRelay (2, 1); { Motor On }
                            end;
                  end;
                  DoorState := dsUp;
                  LastDoorState := dsUp;
                end;
              If IN3 = HighLevel then
                If IN4 = HighLevel then { No Switch }
                  begin
                    If IN2 = HighLevel then LastDoorState := dsDown;
                      Case LastDoorState of
                        dsUp: If IN2 = LowLevel then
                                begin { now Down }
                                  Case ModuleType of
                                    mtCrawford: begin
                                                  LightOnAndDelay;
                                                  RelayTimer [3] := RelayTime [1].W; { Down }
                                                  SetRelay (3, 1);
                                                end;
                                    mtDoor: begin
                                              SetRelay (1, MotorDown); { Direction Down }
                                              SetRelay (3, 1); { Lock Off }
                                              DelayRelayTime1;
                                              SetRelay (2, 1); { Motor On }
                                            end;
                                  end;
                                  DoorState := dsDown;
                                  LastDoorState := dsDown;
                                end;
                        else begin
                               Case ModuleType of
                                 mtCrawford: begin
                                               LightOnAndDelay;
                                               RelayTimer [1] := RelayTime [1].W; { Up }
                                               SetRelay (1, 1);
                                             end;
                                 mtDoor: begin
                                           SetRelay (1, MotorUp); { Direction Up }
                                           SetRelay (3, 1); { Lock Off }
                                           DelayRelayTime1;
                                           SetRelay (2, 1); { Motor On }
                                         end;
                               end;
                               DoorState := dsUp;
                               LastDoorState := dsUp;
                             end;
                      end
                  end;
            end;
    else begin { Up or Down }
           Case ModuleType of
             mtCrawford: begin
                           RelayTimer [2] := RelayTime [1].W; { Stop }
                           SetRelay (2, 1);
                         end;
             mtDoor: DoorMotorOff;
           end;
           DoorState := dsStop;
         end
  end;
  RelayTimer [4] := RelayTime [4].W;
  SetRelay (4, 1);
end;

Procedure SetRelayTimer;
begin
  RelayTimer [TempByte] := RelayTime [TempByte].W;
end;

Procedure ProcessNormalTxKey;
begin
  If HighNib (ChData [TempByte]) = TxKeyBits then
    Case TChType (ChData [TempByte] and $0F) of
      chFollow: begin
                  Case TempByte of
                    1..4: RelayTimer [TempByte] := TempWord.L;
                    else begin
                           Relay5_15 := TempByte;
                           Relay5_15Timer := TempWord.L;
                         end;
                  end;
                  SetRelay (TempByte, 1);
                end;
      chToggle: begin
                  If RPT then Exit;
                  SetRelay (TempByte, 2);
                end;
      chPulse: If TempByte <= 4 then
                Case Boolean (Relay [TempByte]) of
                  Relay_OFF: If not RPT then
                               begin
                                 SetRelayTimer;
                                 SetRelay (TempByte, 1);
                               end;
                  Relay_ON: begin
                              If RPT then SetRelayTimer;
                              SetRelay (TempByte, Byte (RPT));
                            end;
                end;
    end;
end;

Procedure ProcessBlinds12State;
begin
  If RPT then Exit;
  Case Blinds12State of
    bsIdle:        begin
                     SetRelay (2, 0);   { Just in case }
                     TempByte := 1;
                     SetRelayTimer;
                     SetRelay (1, 1);
                     Blinds12State := bsOut1;
                   end;
    bsOut1:        begin
                     SetRelay (1, 0);
                     Blinds12State := bsIdleBetween;
                   end;
    bsIdleBetween: begin
                     SetRelay (1, 0);   { Just in case }
                     TempByte := 2;
                     SetRelayTimer;
                     SetRelay (2, 1);
                     Blinds12State := bsOut2;
                   end;
    bsOut2:        begin
                     SetRelay (2, 0);
                     Blinds12State := bsIdle;
                   end;
  end;
end;

Procedure ProcessBlinds34State;
begin
  If RPT then Exit;
  Case Blinds34State of
    bsIdle:        begin
                     SetRelay (4, 0);   { Just in case }
                     TempByte := 3;
                     SetRelayTimer;
                     SetRelay (3, 1);
                     Blinds34State := bsOut1;
                   end;
    bsOut1:        begin
                     SetRelay (3, 0);
                     Blinds34State := bsIdleBetween;
                   end;
    bsIdleBetween: begin
                     SetRelay (3, 0);   { Just in case }
                     TempByte := 4;
                     SetRelayTimer;
                     SetRelay (4, 1);
                     Blinds34State := bsOut2;
                   end;
    bsOut2:        begin
                     SetRelay (4, 0);
                     Blinds34State := bsIdle;
                   end;
  end;
end;

Procedure ProcessTxKey;

  Procedure DelayRelay2;
  begin
    RelayTimer [2] := RelayTime [1].W;
    Repeat
      WriteDisplay;
    until RelayTimer [2] = 0;
  end;

  Procedure ProcessModuleType;
  begin
    Case ModuleType of
      mtNormal: begin
                  Case mfBlinds12 in ModuleFlags of
                    True: If HighNib (ChData [1]) = TxKeyBits then ProcessBlinds12State;
                    else
                      TempByte := 1;
                      ProcessNormalTxKey;
                      TempByte := 2;
                      ProcessNormalTxKey;
                  end;
                  Case mfBlinds34 in ModuleFlags of
                    True: If HighNib (ChData [3]) = TxKeyBits then ProcessBlinds34State;
                    else
                      TempByte := 3;
                      ProcessNormalTxKey;
                      TempByte := 4;
                      ProcessNormalTxKey;
                  end;
                  For TempByte := 5 to 15 do ProcessNormalTxKey;
                end;

      mtCrawford,
      mtDoor:     begin
                    If HighNib (ChData [1]) <> TxKeyBits then Exit;
                    If RPT then Exit;
                    If NoKeyTimer = 0 then ProcessDoor;
                    NoKeyTimer := NoKeyTime;
                  end;
      mtAlarm: begin
                 For TempByte := 3 to 4 do ProcessNormalTxKey;
                 If HighNib (ChData [1]) <> TxKeyBits then Exit;
                 If RPT then Exit;
                 SetRelay (1, 2);
                 Case Boolean (Relay [1]) of
                   Relay_OFF: begin
                                RelayTimer [2] := RelayTime [2].W;
                                SetRelay (2, 1);
                              end;
                   Relay_ON: begin
                                SetRelay (2, 1);
                                DelayRelay2;
                                DelayRelay2;
                                SetRelay (2, 1);
                                DelayRelay2;
                                DelayRelay2;
                                SetRelay (2, 1);
                                RelayTimer [2] := RelayTime [1].W;
                             end;
                 end;
               end;
    end;
  end;

begin
  Display := HexChar [TxKeyBits];
  Case ModuleType of
    mtDoor: ClsTimer := 600;
    else ClsTimer := 600;
  end;
  If ReadByteFromEEPROM (TempWord.W + Ofs (TxData.tdSerialNumber [3])) and $10 = 0 then
    begin                                        { Not disabled }
      ET0 := False;
      TempWord.L := KeyProcessingTimer div 32;
      ET0 := True;
      Inc (TempWord.L, 10);
      If TxKeyBits <> LastTxKeyBits then
        TempWord.L := 2 * TempWord.L;
      ProcessModuleType;
    end;
  If TxKeyBits <> LastTxKeyBits then
    begin
      If not AuxModule then SendToTerminal;
      DelayTimer := (ModuleAddress and $03) * 10;
      Repeat
        WriteDisplay;
      until DelayTimer = 0;
      Repeat until not Sending;
      TX_Buffer [bpCommand] := Cmd_LNG_Tx;
      TX_Buffer [bpDestinationAddress] := $FF;
      TX_Buffer [bpParameter1] := Counter.H;
      TX_Buffer [bpParameter2] := Counter.L;
      TX_Buffer [bpParameter3] := SerialNumber [3] and $0F or TempWord.H;
      TX_Buffer [bpParameter4] := SerialNumber [2];
      TX_Buffer [bpParameter5] := SerialNumber [1];
      TX_Buffer [bpParameter6] := SerialNumber [0];
      TX_Buffer [bpParameter7] := TxKeyBits;
      TX_Buffer [bpParameter8] := Byte (VLow);
      SendLNGBlock;
    end;
  LastTxKeyBits := TxKeyBits;
end;


Procedure ProcessTxPacket;
Var I: Byte;

  Function CheckPgmKey: Boolean;
  begin
    CheckPgmKey := False;
    If SerialNumber [3] and $0F <> $0B then Exit;
    If SerialNumber [2] <> $EE then Exit;
    If SerialNumber [1] <> $00 then Exit;
    If SerialNumber [0] <> $01 then Exit;
    CheckPgmKey := True;
  end;

  Procedure UpdateCounter;
  begin
    WriteByteToEEPROM (TempWord.W + Ofs (TxData.tdCounter.L), Counter.L);
    WriteByteToEEPROM (TempWord.W + Ofs (TxData.tdCounter.H), Counter.H);
  end;


  Procedure GenerateKeyAndDecrypt;
  begin
    TempBits32 [0] := EncryptedCode [0];
    TempBits32 [1] := EncryptedCode [1];
    TempBits32 [2] := EncryptedCode [2];
    TempBits32 [3] := EncryptedCode [3];
    GenerateKey;
    EncryptedCode [0] := TempBits32 [0];
    EncryptedCode [1] := TempBits32 [1];
    EncryptedCode [2] := TempBits32 [2];
    EncryptedCode [3] := TempBits32 [3];
    SetCalculatedKey;
    Decrypt;
  end;

begin
  TxKeyBits := (DataByte [7] shr 5) and $07;
  If (DataByte [7] and $10) <> 0 then TxKeyBits := TxKeyBits or $08;
  KeyPacketReceived := False;
  If TxKeyBits = 0 then Exit;
  Case Setup of
    None: begin
            If RPT then
              If not TempSet then
                begin
                  If TxKeyBits <> LastTxKeyBits then RPT := False;
                end;
            If Key_Enter = LowLevel then
              begin
                KeyPacketReceived := True;
                Setup := LearnKey;
                TX_Buffer [bpDestinationAddress] := $FF;
                Repeat
                  RestartWatchDog;
                until not Sending;
                SendStatus;
                Exit;
              end;

            If not TXInEEprom then
              Case AutoSaveDisabled of
                True: begin
                        If not AuxModule then
                          begin

                          { Send to terminal if TX is not in EEPROM }

                            GenerateKeyAndDecrypt;
                            If not DecryptOK then Exit;
                            If TxKeyBits <> LastTxKeyBits then SendToTerminal;
                            LastTxKeyBits := TxKeyBits;
                            ClsTimer := 600;                             { Delay before LastTxKeyBits is cleared }
                          end;

                        Exit;
                      end;
                else
                  GenerateKeyAndDecrypt;
                  If not DecryptOK then Exit;
                  If not SaveTxToEEprom then Exit;
              end;

            For I := 0 to 7 do Key [I] := ReadByteFromEEprom (TempWord.W + Ofs (TxData.tdKey) + I);
            Decrypt;
            If not DecryptOK then Exit;
            If Key_Cancel = LowLevel then
              begin
                RemoveTx;
                Exit;
              end;
            If Key_Up = LowLevel then
              begin
                Setup := SetModuleType;
                TempModuleType := ModuleType;
                SetupTimer := SetupTime;
                For I := 1 to 15 do SetRelay (I, 0);
                Exit;
              end;
            If Key_Down = LowLevel then
              begin
                Setup := SetBlindsMode;
                If mfBlinds12 in ModuleFlags then
                  Case mfBlinds34 in ModuleFlags of
                    True: BlindsMode := bmCh1234;
                    else  BlindsMode := bmCh12;
                  end else If mfBlinds34 in ModuleFlags then BlindsMode := bmCh34 else BlindsMode := bmNone;
                TempBlindsMode := BlindsMode;
                SetupTimer := SetupTime;
                Exit;
              end;
            If CheckPgmKey then
              begin
                If RPT then Exit;
                UpdateCounter;
                Setup := LearnKey;
                TX_Buffer [bpDestinationAddress] := $FF;
                Repeat until not Sending;
                SendStatus;
                Exit;
              end;

            If CounterCheck and (SerialNumber [3] and $0F <> $0E) then
              Case TempSet of
                False: begin
                         TempCounter.L := ReadByteFromEEprom (TempWord.W + Ofs (TxData.tdCounter.L));
                         TempCounter.H := ReadByteFromEEprom (TempWord.W + Ofs (TxData.tdCounter.H));

                         CounterDifference.W := Counter.W - TempCounter.W;

                         If (CounterDifference.H <> 0) or not (CounterDifference.L in [0..16]) or ((CounterDifference.L = 0) and not RPT) then
                           begin
                             If CounterDifference.W > $8000 then Exit;
                             TempCounter.W := Counter.W;
                             TempSet := True;
                             Exit;
                           end;
                       end;
                True: begin
                        TempSet := False;
                        If Counter.W - TempCounter.W <> 1 then Exit;
                        RPT := False;
                      end;
              end else TempSet := False;

            UpdateCounter;
            ProcessTxKey;
          end;
    LearnKey: begin
                SetupTimer := SetupTime;
                GenerateKeyAndDecrypt;
                If not DecryptOK then
                  begin
                    Display := 'O';
                    Delay (100);
                    Exit;
                  end;
                If CheckPgmKey then
                  If TXInEEprom then
                    begin
                      If RPT then Exit;
                      UpdateCounter;
                      Setup := None;;
                      TX_Buffer [bpDestinationAddress] := $FF;
                      Repeat until not Sending;
                      SendStatus;
                      Exit;
                    end;
                TempCounter.W := Counter.W;
                Setup := LearnKey2;
              end;
    LearnKey2: begin
                SetupTimer := SetupTime;
                GenerateKeyAndDecrypt;
                If not DecryptOK then
                  begin
                    Display := 'O';
                    Delay (100);
                    Setup := LearnKey;
                    Exit;
                  end;
                If CheckPgmKey then
                  If TXInEEprom then
                    begin
                      If RPT then Exit;
                      UpdateCounter;
                      Setup := None;
                      TX_Buffer [bpDestinationAddress] := $FF;
                      Repeat until not Sending;
                      SendStatus;
                      Exit;
                    end;
                If Counter.W - TempCounter.W > 2 then
                  begin
                    Display := 'C';
                    Delay (100);
                    Setup := LearnKey;
                    Exit;
                  end;
                If SaveTxToEEprom then
                  begin
                    Display := 'P';
                    WriteDisplay;
                    Repeat until not Sending;
                    Delay (5);
                    TX_Buffer [bpDestinationAddress] := $FF;
                    SendStatus;
                    Delay (200);
                    ProcessTxKey;
                    Setup := LearnKey;
                  end else Setup := None;
              end;
    SetTxKey: begin
                SetupTimer := SetupTime;
                Case ModuleType of
                  mtNormal,
                  mtAlarm:  begin
                              ChDataBlock [Channel] := NibToByte (TxKeyBits, ChDataBlock [Channel]);
                              WriteByteToEEPROM (Ofs (TEEprom.EE_ChannelConfig) + Channel , ChDataBlock [Channel]);
                              Setup := SetChannel;
                              Case ModuleType of
                                mtNormal : begin
                                             Channel := (Channel + 1) and $0F;
                                             If AuxModule then If Channel = 15 then
                                               Channel := 0;
                                           end;
                                mtAlarm: begin
                                           Channel := (Channel + 1) and $03;
                                           If Channel = 1 then Channel := 2;
                                         end;
                              end
                            end;
                  mtCrawford,
                  mtDoor: begin
                             ChData [1] := NibToByte (TxKeyBits, ChData [1]);
                             WriteByteToEEPROM (Ofs (TEEprom.EE_ChannelConfig), ChData [1]);
                             Setup := None;
                          end;
                end;
                Display := HexChar [TxKeyBits];
                Delay (300);
                Display := ' ';
                Delay (100);
              end;
  end;
end;

Procedure ProcessCommands;
Var TempByte, I: Byte;
    IW: Word;
begin
  BlockReceived := False;
  Delay (2);
  Case RX_Buffer [bpCommand] of
    Cmd_StatusRequest: begin
                         TX_Buffer [bpDestinationAddress] := RX_Buffer [bpSourceAddress];
                         If RX_Buffer [bpParameter1] = $FF then
                           begin
                             TX_Buffer [bpCommand] := Cmd_LNG_Tx;
                             TempWord.W := Ofs (TEEprom.EE_TXKeyData);
                             IW := NumberOfLearnedKeys;
                             While IW <> 0 do
                               begin
                                 Repeat until not Sending;
                                 TX_Buffer [bpParameter1] := ReadByteFromEEprom (TempWord.W + Ofs (TxData.tdCounter.H));
                                 TX_Buffer [bpParameter2] := ReadByteFromEEprom (TempWord.W + Ofs (TxData.tdCounter.L));
                                 For TempByte := 0 to 3 do
                                   TX_Buffer [bpParameter3 + TempByte] := ReadByteFromEEprom (TempWord.W + Ofs (TxData.tdSerialNumber [3]) - TempByte);
                                 TX_Buffer [bpParameter7] := 0;
                                 TX_Buffer [bpParameter8] := 0;
                                 SendLNGBlock;
                                 Inc (TempWord.W, SizeOf (TxData));
                                 Dec (IW);
                               end;
                           end else SendStatus;
                       end;
    Cmd_ChangeAddress: begin
                         If RX_Buffer [bpParameter1] = RX_Buffer [bpParameter2] then
                           begin
                             ModuleAddress := RX_Buffer [bpParameter1];
                             WriteByteToEEPROM (Ofs (TEEprom.EE_ModuleAddress), ModuleAddress);
                           end;
                       end;
    Cmd_GetData: begin
                   TX_Buffer [bpCommand] := RX_Buffer [bpParameter1];
                   TX_Buffer [bpDestinationAddress] := RX_Buffer [bpSourceAddress];
                   Case RX_Buffer [bpParameter1] of
                     Cmd_SoftwareVersion: begin
                                            TX_Buffer [bpParameter1] := VersionHi;
                                            TX_Buffer [bpParameter2] := VersionLo;
                                            SendBlock;
                                          end;
                     Cmd_RelayTime: begin
                                      TempByte := RX_Buffer [bpParameter2] and $03 + 1;
                                      TX_Buffer [bpParameter1] := RelayTime [TempByte].H and $3F or RX_Buffer [bpParameter2] shl 6;
                                      TX_Buffer [bpParameter2] := RelayTime [TempByte].L;
                                      SendBlock;
                                    end;
                     Cmd_Cfg: begin
                                TX_Buffer [bpParameter1] := Byte (ModuleType);
                                TX_Buffer [bpParameter2] := Byte (ModuleFlags);
                                TX_Buffer [bpParameter11] := RX_Buffer [bpParameter2];
                                If RX_Buffer [bpParameter2] = 0 then
                                  For I := 0 to 7 do TX_Buffer [bpParameter3 + I] := ChDataBlock [I]
                                    else For I := 0 to 6 do TX_Buffer [bpParameter3 + I] := ChDataBlock [I + 8];
                                SendLNGBlock;
                              end;
                   end;
                 end;
    Cmd_Relay: begin
                 If (RX_Buffer [bpParameter2]) and $F0 <> 0 then Exit;
                 If RX_Buffer [bpParameter1] > 2 then Exit;
                 SetRelay (RX_Buffer [bpParameter2], RX_Buffer [bpParameter1]);
               end;
    Cmd_ActivateOutput: begin
                          If (RX_Buffer [bpParameter1] - 1) and $FC <> 0 then Exit;
                          RelayTimer [RX_Buffer [bpParameter1]] := RelayTime [RX_Buffer [bpParameter1]].W;
                          SetRelay (RX_Buffer [bpParameter1], 1);
                        end;
    Cmd_RelayTime: begin
                     TempByte := (RX_Buffer [bpParameter1] shr 6) and $03 + 1;
                     TempWord.H := RX_Buffer [bpParameter1] and $3F;
                     TempWord.L := RX_Buffer [bpParameter2];
                     RelayTime [TempByte].W := TempWord.W;
                     If TempByte = 4 then WriteRelayTimesToEEPROM;
                   end;
    Cmd_LearnLNG_Tx: begin
                       If Setup = None then
                         begin
                           Setup := LearnKey;
                           SetupTimer := SetupTime;
                         end else
                           begin
                             Setup := None;
                             SetupTimer := 0;
                           end;
                       TX_Buffer [bpDestinationAddress] := RX_Buffer [bpSourceAddress];
                       SendStatus;
                     end;
    Cmd_LNG_Tx: begin
                  SerialNumber [3] := RX_Buffer [bpParameter3];
                  SerialNumber [2] := RX_Buffer [bpParameter4];
                  SerialNumber [1] := RX_Buffer [bpParameter5];
                  SerialNumber [0] := RX_Buffer [bpParameter6];
                  If RX_Buffer [bpParameter8] and $02 <> 0 then
                    begin
                      RemoveTx;
                      Exit;
                    end;
                  If not TXInEEprom then Exit;
                  If RX_Buffer [bpParameter8] and $04 <> 0 then
                    begin
                      WriteByteToEEPROM (TempWord.W + Ofs (TxData.tdSerialNumber [3]), RX_Buffer [bpParameter3]);
                      Exit;
                    end;
                  If Counter.L = RX_Buffer [bpParameter2] then
                    If Counter.H = RX_Buffer [bpParameter1] then Exit;
                  WriteByteToEEPROM (TempWord.W + Ofs (TxData.tdCounter.L), RX_Buffer [bpParameter2]);
                  WriteByteToEEPROM (TempWord.W + Ofs (TxData.tdCounter.H), RX_Buffer [bpParameter1]);
                end;
    Cmd_Cfg: begin
               ModuleType := TModuleType (RX_Buffer [bpParameter1]);
               Byte (ModuleFlags) := RX_Buffer [bpParameter2];
               If RX_Buffer [bpParameter11] = 0 then
                 For I := 0 to 7 do
                   begin
                     ChDataBlock [I] := RX_Buffer [bpParameter3 + I];
                     SetRelay (I + 1, 0);
                   end else begin
                              For I := 0 to 6 do
                                begin
                                  ChDataBlock [I + 8] := RX_Buffer [bpParameter3 + I];
                                  SetRelay (I + 9, 0);
                                end;
                              WriteModuleConfig;
                            end;
             end;
  end;
end;

Procedure ProcessSetup;
Const ModuleTypeChar: Array [mtNormal..mtAlarm] of Char = 'NCDA';
      ChannelTypeChar: Array [chFollow..chPulse] of Char = 'FTP';
begin
  Case Setup of
    None: begin
            SetupTimer := 0;
            If ClsTimer = 0 then
              begin
                Display := ' ';
                LastTxKeyBits := 0;
              end;
          end;
    LearnKey: begin
                Display := '-';
                Case Kbd of
                  kCancel: begin
                             Setup := None;
                             TX_Buffer [bpDestinationAddress] := $FF;
                             SendStatus;
                           end;
                end;
              end;
    LearnKey2: begin
                 Display := '=';
                 Case Kbd of
                   kCancel: begin
                              Setup := None;
                              TX_Buffer [bpDestinationAddress] := $FF;
                              SendStatus;
                            end;
                 end;
               end;
    SetModuleType: begin
                     Case Kbd of
                       kUp: begin
                              TempModuleType := Succ (TempModuleType);
                              If TempModuleType = Succ (High (TModuleType)) then TempModuleType := Low (TModuleType);
                            end;
                       kDown: begin
                                TempModuleType := Pred (TempModuleType);
                                If TempModuleType = Pred (Low (TModuleType)) then TempModuleType := High (TModuleType);
                              end;
                       kCancel: Setup := None;
                       kEnter: begin
                                 If ModuleType <> TempModuleType then
                                   begin
                                     ModuleType := TempModuleType;
                                     ModuleFlags := [];
                                     Case ModuleType of
                                       mtNormal: InitNormalModuleType;
                                       mtCrawford,
                                       mtDoor:     begin
                                                     RelayTime [1].W := DefaultRelayShortPulseTime;
                                                     RelayTime [2].W := DefaultRelayShortPulseTime;
                                                     RelayTime [3].W := DefaultRelayShortPulseTime;
                                                     RelayTime [4].W := DafaultLightPulseTime;
                                                     WriteRelayTimesToEEPROM;
                                                     ChData [4] := Byte (chPulse);
                                                     Case ModuleType of
                                                       mtCrawford: begin
                                                                     ChData [1] := Byte (chPulse);
                                                                     ChData [2] := Byte (chPulse);
                                                                     ChData [3] := Byte (chPulse);
                                                                   end;
                                                       mtDoor: begin
                                                                 ChData [1] := Byte (chToggle);
                                                                 ChData [2] := Byte (chToggle);
                                                                 ChData [3] := Byte (chToggle);
                                                               end;
                                                     end
                                                   end;
                                       mtAlarm: begin
                                                  RelayTime [1].W := DefaultRelayShortPulseTime;
                                                  RelayTime [2].W := DefaultRelayOffPulseTime;
                                                  RelayTime [3].W := DafaultRelayLongPulseTime;
                                                  RelayTime [4].W := DafaultRelayLongPulseTime;
                                                  WriteRelayTimesToEEPROM;
                                                  ChData [1] := Byte (chToggle);
                                                  ChData [2] := Byte (chPulse);
                                                  ChData [3] := Byte (chFollow);
                                                  ChData [4] := Byte (chFollow);
                                                end;
                                     end;
                                     WriteModuleConfig;
                                   end;
                                 Setup := SetChannel;
                                 Channel := 0;
                               end;
                     end;
                     Display := ModuleTypeChar [TempModuleType];
                     If Kbd <> kNone then SetupTimer := SetupTime;
                   end;
    SetChannel: begin
                  Case ModuleType of
                    mtNormal: begin
                                Case Kbd of
                                  kUp: begin
                                         Channel := (Channel + 1) and Mask;
                                         If AuxModule then If Channel = 15 then
                                           Channel := 0;
                                       end;
                                  kDown: begin
                                           Channel := (Channel - 1) and Mask;
                                           If AuxModule then If Channel = 15 then
                                             Channel := 14;
                                         end;
                                  kCancel: begin
                                             Setup := SetModuleType;
                                             TempModuleType := ModuleType;
                                           end;
                                  kEnter: begin
                                            Setup := SetChannelType;
                                            TempChannelType := TChType (ChDataBlock [Channel] and $0F);
                                          end;
                                end;
                                Display := HexChar [Channel + 1];
                              end;
                    mtCrawford,
                    mtDoor: begin
                              Setup := SetTxKey;
                              Display := '=';
                            end;
                    mtAlarm: begin
                               Case Kbd of
                                 kUp: begin
                                        Channel := (Channel + 1) and $03;
                                        If Channel = 1 then Channel := 2;
                                      end;
                                 kDown: begin
                                          Channel := (Channel - 1) and $03;
                                          If Channel = 1 then Channel := 0;
                                        end;
                                 kCancel: begin
                                            Setup := SetModuleType;
                                            TempModuleType := ModuleType;
                                          end;
                                 kEnter: begin
                                           If Channel = 0 then
                                              begin
                                                Setup := SetTxKey;
                                                Display := '=';
                                                Exit;
                                              end else begin
                                                         Setup := SetChannelType;
                                                         TempChannelType := TChType (ChDataBlock [Channel] and $0F);
                                                       end
                                         end;
                               end;
                               Display := Char (Channel + $31);
                             end;
                  end;
                  If Kbd <> kNone then SetupTimer := SetupTime;
                end;
    SetChannelType: begin
                      Case ModuleType of
                        mtNormal,
                        mtAlarm:  begin
                                    Case Kbd of
                                      kUp: begin
                                             If TempChannelType = chPulse then TempChannelType := chFollow
                                               else TempChannelType := TChType (Ord (TempChannelType) + 1);
                                           end;
                                      kDown: begin
                                               If TempChannelType = chFollow then TempChannelType := chPulse
                                                 else TempChannelType := TChType (Ord (TempChannelType) - 1);
                                               end;
                                      kCancel: begin
                                                 Setup := SetChannel;
                                               end;
                                      kEnter: begin
                                                ChDataBlock [Channel] := NibToByte (HighNib (ChDataBlock [Channel]), Byte (TempChannelType));
                                                WriteByteToEEPROM (Ofs (TEEprom.EE_ChannelConfig) + Channel , ChDataBlock [Channel]);
                                                Case (Channel and $FC = 0) and (TempChannelType = chPulse) of
                                                  True: begin
                                                          Setup := SetTime;
                                                          TempTime := RelayTime [Channel + 1].W div (10000 div 32);
                                                        end;
                                                  else
                                                    Setup := SetTxKey;
                                                    Display := '=';
                                                end;
                                                Exit;
                                              end;
                                    end;
                                    Display := ChannelTypeChar [TempChannelType];
                                  end;
                      end;
                      If Kbd <> kNone then SetupTimer := SetupTime;
                    end;
    SetTime:  begin
                Case ModuleType of
                  mtNormal,
                  mtAlarm:  begin
                              Case Kbd of
                                kUp: begin
                                       TempTime := (TempTime + 1) and $0F;
                                     end;
                                kDown: begin
                                         TempTime := (TempTime - 1) and $0F;
                                       end;
                                kCancel: begin
                                           Setup := SetChannelType;
                                         end;
                                kEnter: begin
                                          Case TempTime of
                                            0:   RelayTime [Channel + 1].W := 5050 div 32;
                                            else RelayTime [Channel + 1].W := TempTime * (10050 div 32);
                                          end;

                                          WriteByteToEEPROM (Ofs (TEEprom.EE_RelayTime) + Channel * 2 + 0, RelayTime [Channel + 1].L);
                                          WriteByteToEEPROM (Ofs (TEEprom.EE_RelayTime) + Channel * 2 + 1, RelayTime [Channel + 1].H);
                                          Setup := SetTxKey;
                                          Display := '=';
                                          Exit;
                                        end;
                              end;
                              If BlinkTimer.H and $2 = 0 then Display := ' ' else Display := HexChar [TempTime];
                            end;
                end;
                If Kbd <> kNone then SetupTimer := SetupTime;
              end;
    SetTxKey: begin
                Case Kbd of
                  kCancel: Case ModuleType of
                             mtNormal: Setup := SetChannelType;
                             mtAlarm: Case Channel of
                                        0: Setup := SetModuleType;
                                        else Setup := SetChannelType;
                                      end;
                             else Setup := SetModuleType;
                           end;
                  kEnter: begin
                            Display := '0';
                            WriteDisplay;
                            ChDataBlock [Channel] := ChDataBlock [Channel] and $0F;
                            WriteByteToEEPROM (Ofs (TEEprom.EE_ChannelConfig) + Channel, ChDataBlock [Channel]);
                            Setup := SetChannel;
                            Channel := (Channel + 1) and Mask;
                            If ModuleType =  mtAlarm then
                              If Channel = 1 then Channel := 2;
                            If AuxModule then If Channel = 15 then
                              Channel := 0;
                            Delay (300);
                            Display := ' ';
                            Delay (100);
                          end;
                end;
              end;
    SetBlindsMode: begin
                     Case ModuleType of
                       mtNormal: begin
                                   Case Kbd of
                                     kUp: begin
                                            TempBlindsMode := Succ (TempBlindsMode);
                                            If TempBlindsMode = Succ (High (TBlindsMode)) then TempBlindsMode := Low (TBlindsMode);
                                          end;
                                     kDown: begin
                                            TempBlindsMode := Pred (TempBlindsMode);
                                            If TempBlindsMode = Pred (Low (TBlindsMode)) then TempBlindsMode := High (TBlindsMode);
                                              end;
                                     kCancel: begin
                                                Setup := None;
                                              end;
                                     kEnter:  begin
                                                ModuleFlags := ModuleFlags - [mfBlinds12, mfBlinds34];
                                                Case TempBlindsMode of
                                                  bmCh12:   ModuleFlags := ModuleFlags + [mfBlinds12];
                                                  bmCh34:   ModuleFlags := ModuleFlags + [mfBlinds34];
                                                  bmCh1234: ModuleFlags := ModuleFlags + [mfBlinds12, mfBlinds34];
                                                end;
                                                WriteByteToEEPROM (Ofs (TEEprom.EE_ModuleFlags), Byte (ModuleFlags));
                                                Setup := None;
                                                Exit;
                                              end;
                                   end;
                                   If BlinkTimer.H and $2 = 0 then Display := '/' else
                                     Case TempBlindsMode of
                                       bmNone:   Display := '/';
                                       bmCh12:   Display := 'E';
                                       bmCh34:   Display := '3';
                                       bmCh1234: Display := '8';
                                     end;
                                 end;
                       else Setup := None;           { Blinds are allowed only in normal mode }
                     end;
                     If (Kbd <> kNone) and (Setup <> None) then SetupTimer := SetupTime;
                   end;
  end;
end;

Procedure ReadKeyboard;
Var TempKey: TKbd;
begin
  KeyBits := P2 and $0F;
  Kbd := kNone;
  If KeyBits = LastKeyBits then Exit;
  For TempKey := kNone to kDown do
    If KeyBits = KeyTable [TempKey] then
      begin
        Kbd := TempKey;
        LastKeyBits := KeyBits;
      end;
end;

Procedure ProcessInputs;
Var PosEdge1: Boolean;
    PosEdge2: Boolean;
    PosEdge3: Boolean;
    PosEdge4: Boolean;

  Procedure CheckDoorInputs;
  begin
    If IN2 then If DoorState = dsDown then { Door goes down and light sensor interrupted }
      begin
        Case ModuleType of
          mtCrawford: begin
                        RelayTimer [2] := RelayTime [1].W; { Stop }
                        SetRelay (2, 1);
                        LightOnAndDelay;
                        DelayRelayTime1;
                        DelayRelayTime1;
                        RelayTimer [1] := RelayTime [1].W; { Up }
                        SetRelay (1, 1);
                      end;
          mtDoor: begin
                    SetRelay (2, 0); { Motor Off }
                    DelayRelayTime2;
                    SetRelay (1, MotorUp); { Direction Up }
                    DelayRelayTime2;
                    SetRelay (2, 1); { Motor On }
                  end;
        end;
        DoorState := dsUp;
        LastDoorState := dsUp;
        Exit;
      end;
    If PosEdge4 or PosEdge3 then
      begin
        DoorState := dsStop;
        If ModuleType = mtDoor then DoorMotorOff;
      end;
    If PosEdge1 and (NoKeyTimer = 0 ) then ProcessDoor;
  end;

begin
  Input1 := not IN1;
  Input2 := not IN2;
  Input3 := not IN3;
  Input4 := not IN4;
  PosEdge1 := Input1 and not LastInput1;
  PosEdge2 := Input2 and not LastInput2;
  PosEdge3 := Input3 and not LastInput3;
  PosEdge4 := Input4 and not LastInput4;
  Case ModuleType of
    mtNormal: begin
                If PosEdge1 then
                  begin
                    Case mfBlinds12 in ModuleFlags of
                      True: ProcessBlinds12State;
                      else
                        SetRelay (1, 0);
                        LastTxKeyBits := 0;
                    end;
                  end;
                If PosEdge2 then
                  begin
                    SetRelay (2, 0);
                    LastTxKeyBits := 0;
                  end;
                If PosEdge3 then
                  begin
                    Case mfBlinds34 in ModuleFlags of
                      True: ProcessBlinds34State;
                      else
                        SetRelay (3, 0);
                        LastTxKeyBits := 0;
                    end;
                  end;
                If PosEdge4 then
                  begin
                    SetRelay (4, 0);
                    LastTxKeyBits := 0;
                  end;
              end;
    mtCrawford,
    mtDoor: CheckDoorInputs;
  end;
  If Input1 then NoKeyTimer := NoKeyTime;
  LastInput1 := Input1;
  LastInput2 := Input2;
  LastInput3 := Input3;
  LastInput4 := Input4;
end;

begin
  Init;
  Repeat
    ReadKeyboard;
    ProcessSetup;
    If NewBit then StoreBit;
    If KeyPacketReceived then ProcessTxPacket;
    If BlockReceived then ProcessCommands;
    ProcessInputs;
    WriteDisplay;
    If AuxModule then SendToAuxModule;
  until False;
end.
